export interface IoTCatalogueServiceConstructor {
    token: string,
    baseURL: string
}

export interface URLManagerRequest<T = any> {
    id: string
    json: T,
}

interface AssetListIframeRequest {
    assetIds: string[],
    origin?: string,
    itemsPerPage?: number
}

interface AssetDetailsIframeRequest {
    assetId: string,
    origin?: string
}

interface TrainingResourceDetailsIframeRequest {
    trainingResourceId: string,
    origin?: string
}

interface ValidationDetailsIframeRequest {
    validationId: string,
    origin?: string
}

interface TemporaryIframeRequest {
    assetIds?: string[],
    trainingResourceIds?: string[],
    validationIds?: string[],
    path?: string,
    origin?: string,
    requests?: URLManagerRequest[]
}

export interface TemporaryIframe {
    token: string,
    url: string
}

export class IoTCatalogueService {
    private token: string
    private baseURL: string
    constructor({ baseURL, token }: IoTCatalogueServiceConstructor) {
        this.token = token
        this.baseURL = baseURL
    }

    async getCount(collectionName: string, tagNames?: string[]): Promise<number> {
        if (!collectionName) return 0
        let url = `${this.baseURL}/api/count/${collectionName}`
        if (tagNames) url = `${url}?includeTagNames=${tagNames.join(",")}`
        const response = await fetch(url, {
            method: "GET",
            headers: {
                "Authorization": `Bearer ${this.token}`
            }
        })
        const json = await response.json()
        return json || 0
    }

    private async generateTemporaryIframe({ path, assetIds, origin, requests, trainingResourceIds, validationIds }: TemporaryIframeRequest): Promise<TemporaryIframe | null> {
        const data: any = {}
        if (assetIds) data.components = { ids: assetIds }
        if (trainingResourceIds) data.trainingResources = { ids: trainingResourceIds }
        if (validationIds) data.validations = { ids: validationIds }
        const request = {
            origin,
            path,
            data,
            setDataPublic: true,
            requests
        }
        const body = JSON.stringify(request)
        const headers = {
            "Content-Type": "application/json",
            "Authorization": `Bearer ${this.token}`
        }
        const url = `${this.baseURL}/api/workspaces/generateTemporaryWorkspace`
        const response = await fetch(url, { headers, body: body, method: "POST" })
        const json = await response.json()
        return json || null
    }

    private async generateTemporaryIframeURL(request: TemporaryIframeRequest): Promise<string | null> {
        return (await this.generateTemporaryIframe(request))?.url || null
    }

    async searchAssets(assetIds: string[], request: any): Promise<null | any> {
        const { token } = await this.generateTemporaryIframe({ assetIds }) || {}
        if (!token) return null
        const headers = {
            "Content-Type": "application/json",
            "Authorization": `Bearer ${token}`
        }
        const body = JSON.stringify(request)
        const url = `${this.baseURL}/api/data/search`
        const response = await fetch(url, { headers, body: body, method: "POST" })
        const json = await response.json()
        return json || null
    }

    async generateAssetListIframe({ assetIds, itemsPerPage, origin }: AssetListIframeRequest): Promise<null | string> {
        if (!assetIds) return null
        const path = "/embedded/componentListNoBar"
        let requests: URLManagerRequest[] | undefined
        if (itemsPerPage) requests = [{ id: "componentList", json: { itemsPerPage } }]
        return this.generateTemporaryIframeURL({ path, assetIds, origin, requests })

    }

    private async getElementIds(collectionName: string): Promise<null | string[]> {
        const response = await this.getDataElementsId(collectionName)
        if (!response) return null
        return response.map(e => e._id)
    }

    private getTrainingResourceIds(): Promise<null | string[]> {
        return this.getElementIds("trainingResources")
    }

    private getValidationIds(): Promise<null | string[]> {
        return this.getElementIds("validations")
    }

    async generateTrainingResourceListIframe(origin?: string, urlManagerRequests: URLManagerRequest[] = []): Promise<null | string> {
        const path = "/embedded/trainingResources"
        const trainingResourceIds = await this.getTrainingResourceIds()
        if (trainingResourceIds === null) return null
        const requests: URLManagerRequest[] = [
            {
                id: "trainingPage",
                json: { hideHeader: true }
            },
            ...urlManagerRequests
        ]
        return this.generateTemporaryIframeURL({ path, origin, trainingResourceIds, requests })
    }

    async generateAssetIframeDetails({ assetId, origin }: AssetDetailsIframeRequest): Promise<null | string> {
        const path = `/embedded/components/${assetId}`
        return this.generateTemporaryIframeURL({ path, assetIds: [assetId], origin })
    }

    async generateTrainingResourceIframeDetails({ trainingResourceId, origin }: TrainingResourceDetailsIframeRequest): Promise<null | string> {
        const ids = await this.getTrainingResourceIds()
        if (ids === null || !ids.includes(trainingResourceId)) return null
        const path = `/embedded/trainingResources/${trainingResourceId}`

        return this.generateTemporaryIframeURL({ path, trainingResourceIds: [trainingResourceId], origin })
    }

    async generateValidationIframeDetails({ validationId, origin }: ValidationDetailsIframeRequest): Promise<null | string> {
        const ids = await this.getValidationIds()
        if (ids === null || !ids.includes(validationId)) return null
        const path = `/embedded/validations/${validationId}`
        return this.generateTemporaryIframeURL({ path, validationIds: [validationId], origin })
    }

    async getDataElementsId(pageName: string): Promise<any> {
        const headers = {
            "Content-Type": "application/json",
            "Authorization": `Bearer ${this.token}`
        }
        const url = `${this.baseURL}/api/getTPIElementsId?pageName=${pageName}`
        const requestOptions: RequestInit = {
            method: 'GET',
            headers,
            redirect: 'follow'
        }
        const response = await fetch(url, requestOptions)
        return response.json()
    }

    async getDataElement(pageName: string, id: string, extraFields?: string[]): Promise<any> {
        const headers = {
            "Content-Type": "application/json",
            "Authorization": `Bearer ${this.token}`
        }
        let url = `${this.baseURL}/api/getTPIElement?pageName=${pageName}&id=${id}`
        if (extraFields) url = `${url}&extraFields=${extraFields.join(",")}`
        const requestOptions: RequestInit = {
            method: 'GET',
            headers,
            redirect: 'follow'
        }
        const response = await fetch(url, requestOptions)
        return response.json()
    }
}